/********************************************************************************/
/*Ce logiciel permet de :														*/
/*  	# piloter Robotino ;													*/
/*		# obtenir l'information associe aux diffrents capteurs ;				*/
/*		# lancer diffrents algorithmes de dtection d'obstacle 				*/
/* 			ou de suivi de trajectoire ;										*/
/* 		# traiter la vido du Robotino (cartomtre + filtres de dtection		*/
/*			de contour).														*/
/* 																				*/
/* Copyright (C) <2012>  <Franck RIVIER>										*/	
/*																				*/
/*This file is part of RobotinoSti2d.                                           */
/*                                                                              */
/*RobotinoSti2d is free software: you can redistribute it and/or modify			*/
/*it under the terms of the GNU General Public License as published by			*/
/*the Free Software Foundation, either version 3 of the License, or		 		*/
/*(at your option) any later version.											*/
/*																				*/
/*RobotinoSti2d is distributed in the hope that it will be useful,				*/
/*but WITHOUT ANY WARRANTY; without even the implied warranty of				*/
/*MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the					*/
/*GNU General Public License for more details.									*/
/*																				*/
/*You should have received a copy of the GNU General Public License				*/
/*along with RobotinoSti2d.  If not, see <http://www.gnu.org/licenses/>.		*/
/* 																				*/
/*********************************************************************************/
package robotino;

import java.awt.Image;
import java.util.ArrayList;
import java.util.Collection;
import java.util.concurrent.CopyOnWriteArrayList;
import filtres.EcartoMeter;
import math.Vector;
import math.VectorTransform;
import rec.robotino.com.AnalogInput;
import rec.robotino.com.Bumper;
import rec.robotino.com.Camera;
import rec.robotino.com.Com;
import rec.robotino.com.DistanceSensor;
import rec.robotino.com.Motor;
import rec.robotino.com.OmniDrive;
import rec.robotino.com.PowerManagement;

/**
 * <b>Met en oeuvre les principales classes de l'API OpenRobotino
 * (rec.robotino.com) :</b>
 * <p>
 * <ul>
 * <li>classe Com : connexion avec le robot distant ;</li>
 * <li>classe Motor : acquisition des diffrentes grandeurs de chaque moteur ;</li>
 * <li>classe OmniDrive : pilotage des moteurs  partir d'une commande en x, y
 * et omega ;</li>
 * <li>classe Bumper : dtection d'obstacles par contact ;</li>
 * <li>classe Camra : rception des images de la camra embarque ;</li>
 * <li>classe AnalogInput : capteur de proximit inductif et analogique ;</li>
 * <li>classe DistanceSensor : capteurs de distance analogiques.</li>
 * </ul>
 * </p>
 * <b>Simulation sous RobotinoSimView : </b>
 * <p>
 * <ul>
 * <li>Entre analogique 0 : capteur inductif situ sur l'avant du Robotino</li>
 * <li>Entre analogique 1 : capteur inductif situ sur l'axe de rotation du
 * Robotino</li>
 * <li>Capteur de distance : 0,3 V (distance max) et environ 2,55 V (distance
 * min)</li>
 * </ul>
 * </p>
 * 
 * @author Franck RIVIER
 * 
 */

@SuppressWarnings("unused")
public class Robot {

	protected final AnalogInput analogInput1;
	protected final Bumper bumper;
	protected final Camera camera;
	protected final Com com;
	protected final ArrayList<DistanceSensor> distanceSensors;
	protected final Collection<RobotListener> listeners;
	protected final Motor motor1;
	protected final Motor motor2;
	protected final Motor motor3;
	private Vector obstacle;
	protected final ArrayList<Vector> obstacleVectors;
	protected final OmniDrive omniDrive;
	protected final PowerManagement powerManagement;
	protected final float seuil = 0.04f;

	public Robot() {

		com = new MyCom();
		motor1 = new Motor();
		motor2 = new Motor();
		motor3 = new Motor();
		omniDrive = new OmniDrive();
		bumper = new Bumper();
		camera = new MyCam();
		analogInput1 = new AnalogInput();
		listeners = new CopyOnWriteArrayList<RobotListener>();
		powerManagement = new PowerManagement();
		distanceSensors = new ArrayList<DistanceSensor>();
		for (int i = 0; i < 9; i++) {
			DistanceSensor s = new DistanceSensor();
			s.setSensorNumber(i);
			distanceSensors.add(s);
		}
		obstacleVectors = new ArrayList<Vector>();
		for (int i = 0; i < 9; i++) {
			obstacleVectors.add(new Vector());
		}
		obstacle = new Vector();
		init();
	}

	public void addRobotListener(RobotListener listener) {

		listeners.add(listener);
	}

	/**
	 * Permet de suivre une ligne  partir de l'analyse de l'image fournie par
	 * Robotino.
	 */
	public void cameraLineFollow() {

		// Entrez sur la ligne suivante votre code

	}

	/**
	 * Permet d'viter les obstacles.
	 */
	public void collisionAvoidance() {

		// Entrez sur la ligne suivante votre code

	}

	public void connect(String hostname, boolean block) {
		com.setAddress(hostname);
		com.connect(block);
	}

	public void disconnect() {
		com.disconnect();
	}

	/**
	 * Permet de suivre une ligne au sol  partir des informations fournies par
	 * le capteur inductif.
	 */
	public void inductiveLineFollow() {

		float X, error;		
		X = analogInput1.value();
		error = X - 6;

		// Entrez sur la ligne suivante votre code

	}

	protected void init() {
		motor1.setComId(com.id());
		motor1.setMotorNumber(0);
		motor2.setComId(com.id());
		motor2.setMotorNumber(1);
		motor3.setComId(com.id());
		motor3.setMotorNumber(2);
		omniDrive.setComId(com.id());
		bumper.setComId(com.id());
		camera.setComId(com.id());
		camera.setStreaming(true);
		/*
		 * Capteur inductif raccord sur l'entre analogique 0
		 */
		analogInput1.setInputNumber(0);
		/*
		 * Capteurs IR de distance (9 capteurs rpartis sur le chssis du
		 * Robotino)
		 */
		for (DistanceSensor distSensor : distanceSensors) {
			distSensor.setComId(com.id());
		}
	}

	public boolean isConnected() {
		return com.isConnected();
	}

	/**
	 * Fonction qui permet de stopper Robotino lors d'un contact avec un
	 * obstacle.
	 */
	public void obstacleDetection() {

		// Entrez sur la ligne suivante votre code

	}

	public void setVelocity(float vx, float vy, float omega) {
		omniDrive.setVelocity(vx, vy, omega);
	}

	/**
	 * Permet de suivre un mur en utilisant le produit scalaire du vecteur
	 * obstacle et du vecteur directeur unitaire de l'axe arrire/avant du
	 * Robotino.
	 * <p>
	 * Rglages :
	 * <ul>
	 * <li>sous SIM View : seuil = 0.04f</li>
	 * <li>en situation relle : seuil = 0.4f</li>
	 * </p>
	 */
	public void wallFollow() {

		// Entrez sur la ligne suivante votre code

	}

	class MyCam extends Camera {

		/**
		 * Mthode appele lorsque le Robotino envoi une image.
		 * 
		 * @param img
		 *            : image envoye par le Robotino au format RGB interleaved
		 * @param width
		 *            : largeur de l'image envoye en pixels
		 * @param height
		 *            : hauteur de l'image envoye en pixels
		 * @param numChannels
		 *            : nombre de canaux vido
		 * @param bitPerChannel
		 *            : taille du buffer en octets
		 * @param step
		 *            : nombre d'octets par ligne
		 */
		public void imageReceivedEvent(Image img, long width, long height,
				long numChannels, long bitPerChannel, long step) {
			for (RobotListener listener : listeners) {
				listener.onImageReceived(img, width, height);
			}

		}

	}

	/**
	 * Permet de grer la communication avec Robotino.
	 * 
	 * @author Franck RIVIER
	 * 
	 */
	class MyCom extends Com {

		/**
		 * Mthode appele lorsque le Robotino est connecte.
		 * 
		 * @author Franck RIVIER
		 */
		public void connectedEvent() {
			System.out.println("Connected");
			for (RobotListener listener : listeners) {
				listener.onConnected();
			}
		}

		/**
		 * Mthode appele lorsque le Robotino est dconnecte.
		 * 
		 * @author Franck RIVIER
		 */
		public void connectionClosedEvent() {
			System.out.println("Disconnected");
			for (RobotListener listener : listeners) {
				listener.onDisconnected();
			}
		}

		/**
		 * Mthode appele lorsqu'une erreur se produit ct Robotino.
		 * 
		 * @author Franck RIVIER
		 */
		public void errorEvent(Error e, String errorStr) {
			System.err.println("Erreur: " + e.toString());
			for (RobotListener listener : listeners)
				listener.onError(e);
		}

	}

}
